import { FLV } from '../FLV';
import FLV_Error from '../../errors/FLV_Error';

describe('FLV Class', () => {
  let flv: FLV;

  beforeEach(() => {
    const name = 'TestFLV';
    const labels = ['Low', 'Medium', 'High'];
    const xValues = [0, 1, 2, 3, 4, 5];
    flv = new FLV(name, labels, xValues);
  });

  it('should initialize the xValues correctly', () => {
    expect(flv.GetXvalues()).toEqual([0, 1, 2, 3, 4, 5]);
  });

  it('should initialize the labels correctly', () => {
    expect(flv.GetLabels()).toEqual(['Low', 'Medium', 'High']);
  });

  it('should add values correctly', () => {
    flv.AddValue(1, 'Low', 0.5);
    expect(flv.GetDoM(1)).toEqual({ Low: 0.5, Medium: 0, High: 0 });
  });

  it('should throw an error when adding a value for a non-existent label', () => {
    expect(() => flv.AddValue(1, 'NonExistent', 0.5)).toThrow(
      new FLV_Error('NonExistent is not in FLV'),
    );
  });

  it('should add multiple values for a label correctly', () => {
    flv.AddValues([1, 2], 'Medium', [0.3, 0.7]);
    expect(flv.GetDoM(1)).toEqual({ Low: 0, Medium: 0.3, High: 0 });
    expect(flv.GetDoM(2)).toEqual({ Low: 0, Medium: 0.7, High: 0 });
  });

  it('should add values for multiple labels correctly', () => {
    flv.AddValuesForLabels(
      [1, 2],
      ['Low', 'High'],
      [
        [0.2, 0.4],
        [0.6, 0.8],
      ],
    );
    expect(flv.GetDoM(1)).toEqual({ Low: 0.2, Medium: 0, High: 0.6 });
    expect(flv.GetDoM(2)).toEqual({ Low: 0.4, Medium: 0, High: 0.8 });
  });

  it('should interpolate values correctly when x is not in FLV', () => {
    flv.AddValues([1, 2], 'Low', [1.0, 0.0]);
    flv.AddValues([1, 2], 'High', [0.0, 1.0]);
    const interpolated = flv.GetDoM(1.5);
    expect(interpolated.Low).toBeCloseTo(0.5);
    expect(interpolated.High).toBeCloseTo(0.5);
  });

  it('should throw an error when adding a value for a non-existent label', () => {
    expect(() => flv.AddValue(1, 'NonExistent', 0.5)).toThrow(
      new FLV_Error('NonExistent is not in FLV'),
    );
  });

  it('should throw an error when adding a value for a non-existent x value', () => {
    expect(() => flv.AddValue(10, 'Low', 0.5)).toThrow(
      new FLV_Error('10 is not in the scope of the FLV'),
    );
  });

  it('should throw an error when accessing a non-existent x value', () => {
    expect(() => flv.GetDoM(10)).toThrow(
      new FLV_Error('10 is not in the scope of the FLV'),
    );
  });

  it('should throw an error when x is below the range', () => {
    expect(() => flv.GetDoM(-1)).toThrow(
      new FLV_Error('-1 is not in the scope of the FLV'),
    );
  });

  it('should throw an error when x is above the range', () => {
    expect(() => flv.GetDoM(6)).toThrow(
      new FLV_Error('6 is not in the scope of the FLV'),
    );
  });
});
